package lendables;
/**
 * A concrete realization of Lendable.
 * 
 * This class implements Lendable's two abstract methods 1. checkSelfOut 2.
 * getLateFee
 * 
 * One new instance variable and one new accessor method are added 1. my_artist
 * 2. getArtist()
 */
public class Video extends Lendable {
	public final static int DAYS_TO_BORROW = 2;
	public final static double FIRST_DAY_LATE_FEE = 5.00;
	public final static double ADDITIONAL_DAY_FEE = 1.50;

	/**
	 * Initialize a Video object
	 * 
	 * @param callNumber
	 *            The unique identifier
	 * @param title
	 *            The title of this Lendable
	 */
	public Video(String callNumber, String title) {
		super(title, callNumber, DAYS_TO_BORROW, FIRST_DAY_LATE_FEE, ADDITIONAL_DAY_FEE);
	}

	/**
	 * This method is called during println messages.
	 * 
	 * @return a string representation of this item
	 */
	@Override
	public String toString() {
		return super.toString();
	}

	// Test Tester
	public static void main(String[] args) {
		Video aVideo = new Video("V1", "Matrix");

		System.out.print("Error? ");
		aVideo.checkSelfIn(); // Can't do

		System.out.println(" available true? " + aVideo.isAvailable());
		System.out.println("  overdue false? " + aVideo.isOverdue());
		System.out.println("  borrower null? " + aVideo.getBorrowerID());
		aVideo.checkSelfOut("KIM");
		System.out.println("available false? " + aVideo.isAvailable());
		System.out.println("  overdue false? " + aVideo.isOverdue());
		System.out.println("  current state: " + aVideo.toString());
		System.out.println("   Late fee 0.0? " + aVideo.getLateFee());
		System.out.println("   borrower KIM? " + aVideo.getBorrowerID());
		System.out.println("       call# V1? " + aVideo.getCallNumber());
		System.out.println("   title Matrix? " + aVideo.getTitle());
		System.out.println();
		DayCounter today = new DayCounter();
		System.out.println("due date " + DAYS_TO_BORROW + " days from "
				+ today + "? " + aVideo.getDueDate());

		// Build a small array of Video objects
		Video[] videos = new Video[4];

		videos[0] = aVideo;
		videos[1] = new Video("V2", "Dances With Wolves");
		videos[2] = new Video("V3", "Never Say Never");
		videos[3] = new Video("V4", "Chicken Run");

		// Show the state of all videos in the List
		System.out.println("\nCurrent List of videos: ");
		for (int index = 0; index < videos.length; index++)
			System.out.println(videos[index].toString());

		videos[0].checkSelfIn();
		videos[1].checkSelfOut("JO");
		videos[2].checkSelfOut("JO");
		videos[3].checkSelfOut("JO");

		// Show the state of all videos in the List
		System.out
				.println("\nList of videos after other check-outs and -ins: ");
		for (Video v : videos) {
			System.out.println(v.toString());
		}

		// Test Lateness
		Video overdue = new Video("Q", "Late fee = 5.00");
		overdue.checkSelfOut("Kim");
		DayCounter daysPast = new DayCounter();
		daysPast.adjustDaysBy(-1);
		overdue.setDueDate(daysPast);
		System.out.println("Modified due date: " + overdue.getDueDate());
		System.out.println(FIRST_DAY_LATE_FEE + "? " + overdue.getLateFee());

		daysPast.adjustDaysBy(-1);
		System.out.println("Modified due date: " + overdue.getDueDate());
		System.out.println((FIRST_DAY_LATE_FEE + ADDITIONAL_DAY_FEE) + "? "
				+ overdue.getLateFee());

		daysPast.adjustDaysBy(-1);
		System.out.println("Modified due date: " + overdue.getDueDate());
		System.out.println((FIRST_DAY_LATE_FEE + 2 * ADDITIONAL_DAY_FEE) + "? "
				+ overdue.getLateFee());
	}
}

/*
Expected output from test driver

Error? *ERROR* V1 is not checked out
 available true? true
  overdue false? false
  borrower null? null
available false? false
  overdue false? false
  current state: Video: V1 'Matrix': due 2008/09/02, KIM
   Late fee 0.0? 0.0
   borrower KIM? KIM
       call# V1? V1
   title Matrix? Matrix

due date 2 days from 2008/08/31? 2008/09/02

Current List of videos: 
Video: V1 'Matrix': due 2008/09/02, KIM
Video: V2 'Dances With Wolves': available
Video: V3 'Never Say Never': available
Video: V4 'Chicken Run': available

List of videos after other check-outs and -ins: 
Video: V1 'Matrix': available
Video: V2 'Dances With Wolves': due 2008/09/02, JO
Video: V3 'Never Say Never': due 2008/09/02, JO
Video: V4 'Chicken Run': due 2008/09/02, JO
Modified due date: 2008/08/30
5.0? 5.0
Modified due date: 2008/08/29
6.5? 6.5
Modified due date: 2008/08/28
8.0? 8.0

 */
