package examples;

import java.io.*;
import java.util.Scanner;

/**
 * This class represents a pet survey, keeping track of the number of votes cast
 * for a set of pets. The results are stored in a file.
 * 
 * @author CS209
 * 
 */
public class PetSurvey {

	private String filename;
	private String[] animals = { "dog", "cat", "bird", "fish", "snake", "other" };
	private int[] votes = new int[animals.length];
	private int totalVotes = 0;

	public PetSurvey(String fn) {
		this.filename = fn;
		importResults();
	}

	/**
	 * Read the survey data from the file
	 */
	private void importResults() {
		try {
			BufferedReader input = new BufferedReader(new FileReader(filename));

			String line;
			int i = 0;
			while ((line = input.readLine()) != null) {

				String data[] = line.split(" ");

				String animal = data[0];
				int thisVotes = Integer.parseInt(data[1]);

				animals[i] = animal;
				votes[i] = thisVotes;
				i++;
				totalVotes += thisVotes;
			}

			input.close();
		} catch (FileNotFoundException e) {
			System.out.println(filename
					+ ", containing the survey results, not found");
			e.printStackTrace();
		} catch (IOException e) {
			System.out
					.println("Something went wrong while importing survey results from"
							+ filename);
			e.printStackTrace();
		}

	}

	/**
	 * Store the current voting results into the file
	 */
	public void storeResults() {
		try {
			PrintWriter writer = new PrintWriter(filename);
			// FileWriter writer = new FileWriter(filename);

			for (int i = 0; i < animals.length; i++) {
				// writer.write(animals[i] + " " + votes[i] + "\n");
				writer.println(animals[i] + " " + votes[i]);
			}

			writer.close();

		} catch (IOException e) {
			System.out.println("Error storing survey results to file "
					+ filename);
			e.printStackTrace();
		}

	}

	/**
	 * 
	 * @return the name of the file containing the survey results
	 */
	public String getFilename() {
		return filename;
	}

	/**
	 * 
	 * @return the array of Strings of animal names in the survey
	 */
	public String[] getAnimals() {
		return animals;
	}

	/**
	 * 
	 * @return the array of integers, representing the number of votes for each
	 *         animal.
	 */
	public int[] getVotes() {
		return votes;
	}

	/**
	 * 
	 * @return the number of votes that have been cast.
	 */
	public int getTotalVotes() {
		return totalVotes;
	}

	/**
	 * Casts a vote for the animal
	 * 
	 * @param animal
	 * @return
	 */
	public boolean castVote(String animal) {
		for (int i = 0; i < animals.length; i++) {
			if (animals[i].equals(animal)) {
				votes[i]++;
				return true;
			}
		}
		return false;
	}

	/**
	 * Display the results from the survey in a nicely formatted way.
	 */
	public void displayResults() {
		System.out.println("Animal\tVotes\tPercentage");
		for (int i = 0; i < animals.length; i++) {
			// System.out.print(animals[i] + "\t");
			// System.out.print(votes[i] + "\t");
			double pct = (double) votes[i] / totalVotes * 100;
			// long displayPct = Math.round(pct);
			// System.out.println(displayPct + "%");
			System.out.printf("%-6s%7d%12.2f%%\n", animals[i], votes[i], pct);
		}
	}

	/**
	 * @param args
	 *            not used in this program.
	 */
	public static void main(String[] args) {
		String mySurveyFile = "petSurvey.dat";
		PetSurvey survey = new PetSurvey(mySurveyFile);

		System.out.println("Current Results: ");
		survey.displayResults();

		// Allow User to Vote
		Scanner scanner = new Scanner(System.in);
		System.out
				.print("What animal do you want to vote for as your favorite? (dog, cat, bird, snake, fish, other): ");

		String animalVoted = scanner.nextLine();

		if (!survey.castVote(animalVoted)) {
			System.out.println("I'm sorry.  " + animalVoted
					+ " is not a valid selection.");
			System.out.println("Check your spelling or select other.");
		}

		// Display updated results
		System.out.println("Updated Results: ");
		survey.displayResults();

		survey.storeResults();
	}

}
