package examples;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;

/**
 * Prints information whenever Window event occurs
 * 
 */
public class WindowEventDemo extends JFrame implements WindowListener,
		WindowFocusListener, WindowStateListener {
	final static String newline = "\n";
	final static String space = "    ";
	JTextArea display;

	public WindowEventDemo() {
		// Create and set up the window.
		setTitle("WindowEventDemo");
		setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);

		display = new JTextArea();
		display.setEditable(false);
		JScrollPane scrollPane = new JScrollPane(display);
		scrollPane.setPreferredSize(new Dimension(500, 450));
		add(scrollPane);

		addWindowListener(this);
		addWindowFocusListener(this);
		addWindowStateListener(this);

		// Display the window.
		pack();
		setVisible(true);

		checkWindowManager();
	}

	// Some window managers don't support all window states.
	public void checkWindowManager() {
		Toolkit tk = this.getToolkit();
		if (!(tk.isFrameStateSupported(Frame.ICONIFIED))) {
			displayMessage("Your window manager doesn't support ICONIFIED.");
		}
		if (!(tk.isFrameStateSupported(Frame.MAXIMIZED_VERT))) {
			displayMessage("Your window manager doesn't support MAXIMIZED_VERT.");
		}
		if (!(tk.isFrameStateSupported(Frame.MAXIMIZED_HORIZ))) {
			displayMessage("Your window manager doesn't support MAXIMIZED_HORIZ.");
		}
		if (!(tk.isFrameStateSupported(Frame.MAXIMIZED_BOTH))) {
			displayMessage("Your window manager doesn't support MAXIMIZED_BOTH.");
		} else {
			displayMessage("Your window manager supports MAXIMIZED_BOTH.");
		}
	}

	public void windowClosing(WindowEvent e) {
		displayMessage("WindowListener method called: windowClosing.");

		// A pause so user can see the message before
		// the window actually closes.
		ActionListener task = new ActionListener() {
			boolean alreadyDisposed = false;

			public void actionPerformed(ActionEvent e) {
				if (!alreadyDisposed) {
					alreadyDisposed = true;
					dispose();
				} else { // make sure the program exits
					System.exit(0);
				}
			}
		};
		Timer timer = new Timer(2000, task); // fire every half second
		timer.start();
	}

	public void windowClosed(WindowEvent e) {
		// This will only be seen on standard output.
		displayMessage("WindowListener method called: windowClosed.");
	}

	public void windowOpened(WindowEvent e) {
		displayMessage("WindowListener method called: windowOpened.");
	}

	public void windowIconified(WindowEvent e) {
		displayMessage("WindowListener method called: windowIconified.");
	}

	public void windowDeiconified(WindowEvent e) {
		displayMessage("WindowListener method called: windowDeiconified.");
	}

	public void windowActivated(WindowEvent e) {
		displayMessage("WindowListener method called: windowActivated.");
	}

	public void windowDeactivated(WindowEvent e) {
		displayMessage("WindowListener method called: windowDeactivated.");
	}

	public void windowGainedFocus(WindowEvent e) {
		displayMessage("WindowFocusListener method called: windowGainedFocus.");
	}

	public void windowLostFocus(WindowEvent e) {
		displayMessage("WindowFocusListener method called: windowLostFocus.");
	}

	public void windowStateChanged(WindowEvent e) {
		displayStateMessage(
				"WindowStateListener method called: windowStateChanged.", e);
	}

	void displayMessage(String msg) {
		display.append(msg + newline);
		System.out.println(msg);
	}

	void displayStateMessage(String prefix, WindowEvent e) {
		int state = e.getNewState();
		int oldState = e.getOldState();
		String msg = prefix + newline + space + "New state: "
				+ convertStateToString(state) + newline + space + "Old state: "
				+ convertStateToString(oldState);
		display.append(msg + newline);
		System.out.println(msg);
	}

	/**
	 * Returns the String representation of the state
	 * 
	 * @param state
	 *            the window's state, as defined in Frame
	 * @return
	 */
	String convertStateToString(int state) {
		if (state == Frame.NORMAL) {
			return "NORMAL";
		}
		if ((state & Frame.ICONIFIED) != 0) {
			return "ICONIFIED";
		}
		// MAXIMIZED_BOTH is a concatenation of two bits, so
		// we need to test for an exact match.
		if ((state & Frame.MAXIMIZED_BOTH) == Frame.MAXIMIZED_BOTH) {
			return "MAXIMIZED_BOTH";
		}
		if ((state & Frame.MAXIMIZED_VERT) != 0) {
			return "MAXIMIZED_VERT";
		}
		if ((state & Frame.MAXIMIZED_HORIZ) != 0) {
			return "MAXIMIZED_HORIZ";
		}
		return "UNKNOWN";
	}

	public static void main(String[] args) {
		new WindowEventDemo();
	}
}
