package examples;

import java.io.*;
import java.util.Scanner;

/**
 * This class represents a pet survey, keeping track of the number of votes cast
 * for a set of pets. The results are stored in a file.
 * 
 * This version uses a FileWriter
 * 
 * @author CS209
 * 
 */
public class PetSurvey {

	private String filename;
	private String[] animals = { "dog", "cat", "bird", "fish", "snake", "other" };
	private int[] votes = new int[animals.length];
	private int totalVotes = 0;

	public PetSurvey(String fn) {
		this.filename = fn;
		importResults();
	}

	/**
	 * Read the survey data from the file
	 */
	private void importResults() {
		try {
			BufferedReader input = new BufferedReader(new FileReader(filename));

			String line;
			int i = 0;
			while ((line = input.readLine()) != null) {
				// System.out.println(line);
				String data[] = line.split(" ");

				String animal = data[0];
				int thisVotes = Integer.parseInt(data[1]);

				animals[i] = animal;
				votes[i] = thisVotes;
				i++;
				totalVotes += thisVotes;
			}

			input.close();
		} catch (FileNotFoundException e) {
			System.out.println(filename
					+ ", containing the survey results, not found");
			e.printStackTrace();
		} catch (IOException e) {
			System.out
					.println("Something went wrong while importing survey results from"
							+ filename);
			e.printStackTrace();
		}

	}

	/**
	 * Store the current voting results into the file. Note the format of the
	 * input file.
	 * 
	 * @throws IOException
	 */
	public void storeResults() throws IOException {
		FileWriter out = new FileWriter(filename);
		// loop through the animals
		for (int i = 0; i < animals.length; i++) {
			// write out the animal, space, votes on separate lines
			out.write(animals[i]);
			out.write(" ");
			// Doesn't seem to work correctly if we try to write the integer --
			// not written as text? Maybe written as binary data because from
			// OutputStreamWriter class
			out.write(String.valueOf(votes[i]));
			out.write("\n");
		}
		out.close();

	}

	/**
	 * 
	 * @return the name of the file containing the survey results
	 */
	public String getFilename() {
		return filename;
	}

	/**
	 * 
	 * @return the array of Strings of animal names in the survey
	 */
	public String[] getAnimals() {
		return animals;
	}

	/**
	 * 
	 * @return the array of integers, representing the number of votes for each
	 *         animal.
	 */
	public int[] getVotes() {
		return votes;
	}

	/**
	 * 
	 * @return the number of votes that have been cast.
	 */
	public int getTotalVotes() {
		return totalVotes;
	}

	/**
	 * Casts a vote for the animal
	 * 
	 * @param animal
	 *            -- must be a valid animal
	 */
	public void castVote(String animal) {
		for (int i = 0; i < animals.length; i++) {
			if (animals[i].equals(animal)) {
				votes[i]++;
				totalVotes++;
				return;
			}
		}
	}

	/**
	 * 
	 * @param animal
	 * @return true iff the animal is a valid animal
	 */
	public boolean validAnimal(String animal) {
		for (int i = 0; i < animals.length; i++) {
			if (animals[i].equals(animal)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Display the results from the survey in a nicely formatted way.
	 */
	public void displayResults() {
		System.out.println("Animal\tVotes\tPercentage");
		for (int i = 0; i < animals.length; i++) {
			double pct = (double) votes[i] / totalVotes * 100;
			System.out.printf("%-6s%7d%12.2f%%\n", animals[i], votes[i], pct);
		}
	}

	/**
	 * @param args
	 *            not used in this program.
	 */
	public static void main(String[] args) {
		final String mySurveyFile = "petSurvey.dat";
		PetSurvey survey = new PetSurvey(mySurveyFile);

		System.out.println("Current Results: ");
		survey.displayResults();

		// Allow User to Vote
		Scanner scanner = new Scanner(System.in);
		System.out
				.print("What animal do you want to vote for as your favorite? (dog, cat, bird, snake, fish, other): ");

		String animalVoted = scanner.nextLine();

		if (!survey.validAnimal(animalVoted)) {
			System.out.println("I'm sorry.  " + animalVoted
					+ " is not a valid selection.");
			System.out.println("Check your spelling or select other.");
		} else {
			survey.castVote(animalVoted);
		}

		// Display updated results
		System.out.println("Updated Results: ");
		survey.displayResults();

		try {
			survey.storeResults();
		} catch (IOException e) {
			System.err.println("Error storing results: ");
			e.printStackTrace();
		}
	}

}
