package lendables;
import java.text.NumberFormat;
import java.util.ArrayList;


/**
 * A class that could model a patron in a library.
 * 
 * Each Borrower can hold the collection of borrowed books and other information
 * a librarian should know. You can ask any Borrower object if she or he can
 * borrow another Lendable (see problem specification).
 */
public class Borrower {
	/** The maximum number of books and Borrower can borrow, originally 3. */
	public static final int MAX_BORROWS = 3;
	/**
	 * If a Borrower has more checkSelfOut fees than this, he or she can not
	 * borrow.
	 */
	public static final double MAX_LATE_FEES = 25.00;

	private String my_ID;
	private String my_name;
	private double my_lateFees;
	private ArrayList<Lendable> my_borrowedItems;

	private static NumberFormat our_currencyFormatter = NumberFormat
			.getCurrencyInstance();

	/**
	 * Construct a Borrower with an ID and name.
	 * 
	 * If the ID is "secret", it could be used later as a password to verify a
	 * valid borrower.
	 */
	public Borrower(String ID, String name) {
		my_ID = ID;
		my_name = name;
		my_lateFees = 0.0;
		my_borrowedItems = new ArrayList<Lendable>(MAX_BORROWS);
	}

	/**
	 * Provide access to this Borrowers ID.
	 * 
	 * @return The ID of this Borrower
	 */
	public String getID() {
		return my_ID;
	}

	/**
	 * Provide access to this Borrowers name.
	 * 
	 * @return The ID of this name.
	 */
	public String getName() {
		return my_name;
	}

	/**
	 * Verify the borrower has no overdue books, has fines less than the amount
	 * allowed, and has fewer than maximum number of borrowed (but not overdue)
	 * books.
	 * 
	 * @return true if the Borrower is able to borrow at least one more
	 *         Lendable.
	 */
	public boolean canBorrow(LendableList theLendables) {
		int numberOfOverdues = 0;

		for (Lendable item : my_borrowedItems) {
			String callNumber = item.getCallNumber();
			Lendable borrowedItem = theLendables
					.getLendableWithCallNumber(callNumber);
			if (borrowedItem == null) {
				System.out
						.println("This Borrower could not find " + callNumber);
				return false;
			}
			if (borrowedItem.isOverdue()) {
				numberOfOverdues++;
			}
		}

		return (my_borrowedItems.size() < MAX_BORROWS && numberOfOverdues == 0 && my_lateFees <= MAX_LATE_FEES);
	}

	/**
	 * This method is called during println messages.
	 * 
	 * @return a string representation of this Borrower
	 */
	@Override
	public String toString() {
		String result = "Borrower: " + my_name + ", " + my_ID + ", "
				+ our_currencyFormatter.format(my_lateFees) + "\n[";

		int count = 0;
		for (Lendable item : my_borrowedItems) {
			count++;
			result = result + "\n " + count + ". " + item;
		}
		result += "\n]";

		return result;
	}

	/**
	 * Add to this borrower's list of Lendables.
	 */
	public void borrowLendable(Lendable aLendable) {
		my_borrowedItems.add(aLendable);
	}

	/**
	 * Remove from this borrower's list of Lendables.
	 */
	public boolean returnLendable(Lendable aLendable) {
		return my_borrowedItems.remove(aLendable.getCallNumber());
	}

	/**
	 * Return's late fees this borrower has accrued.
	 */
	public double getLateFees() {
		return my_lateFees;
	}

	/**
	 * Let other objects pay this Borrower's late fees.
	 */
	public void credit(double payment) {
		my_lateFees -= payment;
	}

	/**
	 * Let this or another object add to this Borrower's late fees.
	 */
	public void debit(double additionalLateFees) {
		my_lateFees += additionalLateFees;
	}
}

