package cards;

/**
 * Represents a playing card. Demonstrates use of enumerated types, enum.
 */
public class Card {

	/**
	 * Represents the ranks in a deck of playing cards
	 */
	public enum Rank {
		DEUCE, THREE, FOUR, FIVE, SIX, SEVEN, EIGHT, NINE, TEN, JACK, QUEEN, KING, ACE;
	}

	/**
	 * Represents the suits in a deck of playing cards
	 */
	public enum Suit {
		CLUBS, DIAMONDS, HEARTS, SPADES;
	}

	// a card won't change its rank or suit after it is created
	private final Rank rank;
	private final Suit suit;

	/**
	 * Creates a new card object, with the given rank and suit
	 * 
	 * @param rank
	 * @param suit
	 */
	public Card(Rank rank, Suit suit) {
		this.rank = rank;
		this.suit = suit;
	}

	/**
	 * Returns this card's rank
	 * 
	 * @return card's rank
	 */
	public Rank rank() {
		return rank;
	}

	/**
	 * Returns this card's suit
	 * 
	 * @return card's suit
	 */
	public Suit suit() {
		return suit;
	}

	/**
	 * 10, J, Q, K: 10 points A: 15 points all others: 5 points
	 * 
	 * @return the value of the card in the game of Rummy
	 */
	public int getRummyValue() {
		switch( rank ) {
		case ACE:
			return 15;
		case TEN:
		case JACK:
		case QUEEN:
		case KING:
			return 10;
		default:
			return 5;
		}
	}

	/**
	 * Determines if this card and another card have the same suit. Returns true
	 * if they do.
	 * 
	 * @param c
	 *            another Card to compare
	 * @return true iff the cards have the same suit
	 */
	public boolean sameSuit(Card c) {
		return this.suit.equals(c.suit);
		// return this.suit == c.suit;
		// return this.suit().equals(c.suit());
	}

	/**
	 * Returns a string representation of this card.
	 * 
	 * @return string representation of a Card in the form &lt;rank&gt; of
	 *         &lt;suit&gt;
	 */
	@Override
	public String toString() {
		// leverages toString() methods of Rank and Suit
		return rank + " of " + suit;
	}

	public static void main(String args[]) {
		Card jackOfDiamonds = new Card(Rank.JACK, Suit.DIAMONDS);
		Card aceOfDiamonds = new Card(Rank.ACE, Suit.DIAMONDS);
		System.out.println(jackOfDiamonds);

		if (jackOfDiamonds.sameSuit(aceOfDiamonds)) {
			System.out.println(jackOfDiamonds + " and " + aceOfDiamonds
					+ " are the same suit.");
		}
		System.out.println("The rummyValue of " + jackOfDiamonds + " is "
				+ jackOfDiamonds.getRummyValue());
	}
}